#!/usr/bin/env python3
"""
Simple Example: Enhanced Video Recording

A basic example showing how to create videos with integrated audio
from team communications in the cooperative command game.
"""

from eval import MultiProviderEvaluator

def main():
    print("🎥 Enhanced Video Recording Example")
    print("=" * 40)
    
    # Create evaluator with enhanced video features
    evaluator = MultiProviderEvaluator(
        difficulty="normal",           # Game difficulty
        seed_index=0,                 # Reproducible seed
        max_rounds=10,                # Short evaluation for demo
        api_provider="qwen",          # Use Qwen model
        input_mode="video",           # Video input mode
        save_media=True,              # Save all media files
        enhanced_video=True,          # 🎵 Enable enhanced video with audio!
        video_fps=0.5,               # 0.5 FPS = 2 seconds per frame
        audio_duration_per_frame=3.0  # 3 seconds of audio per frame
    )
    
    print(f"📋 Configuration:")
    print(f"  - Enhanced Video: {evaluator.enhanced_video}")
    print(f"  - Video FPS: {evaluator.video_fps}")
    print(f"  - Audio Duration: {evaluator.audio_duration_per_frame}s per frame")
    print(f"  - Output Directory: {evaluator.output_dir}")
    
    try:
        print(f"\n🚀 Starting enhanced video evaluation...")
        
        # Run the evaluation
        results = evaluator.run_evaluation()
        
        # Show results
        final_score = results['final_stats']['final_score_normalized']
        steps = results['final_stats']['total_steps']
        
        print(f"\n✅ Evaluation completed!")
        print(f"📊 Final Score: {final_score:.1f}/100")
        print(f"🎮 Total Steps: {steps}")
        
        # Save results
        results_file = evaluator.save_results()
        print(f"📄 Results saved: {results_file}")
        
        # Show generated files
        if evaluator.save_media:
            print(f"\n🎬 Enhanced video files created in: {evaluator.videos_dir}")
            
            # List video files
            video_files = list(evaluator.videos_dir.glob("*.mp4"))
            for video_file in video_files:
                size_mb = video_file.stat().st_size / (1024 * 1024)
                print(f"  📹 {video_file.name} ({size_mb:.1f} MB)")
            
            # List metadata files
            json_files = list(evaluator.videos_dir.glob("*.json"))
            for json_file in json_files:
                if "enhanced" in json_file.name:
                    print(f"  📋 {json_file.name} (enhanced metadata)")
        
        print(f"\n🎉 Success! Your enhanced video with audio is ready!")
        print(f"🔍 Check the output directory to see your video with integrated team communications.")
        
    except Exception as e:
        print(f"\n❌ Error during evaluation: {e}")
        print(f"💡 Make sure you have the required dependencies:")
        print(f"   pip install moviepy opencv-python pillow numpy")
        
    finally:
        # Clean up
        evaluator.close()

if __name__ == "__main__":
    main() 